# -*- coding: UTF-8 -*-
# Uklonjen from __future__ import absolute_import jer je suvišan u Pythonu 3.9+
import os

# Zamijenjen 'import urllib' sa standardnim Py3 modulima
import urllib.request # Za otvaranje URL-a
import urllib.parse # Za urlencode
import re
from ..utilities import log, languageTranslate


def search_subtitles(file_original_path, title, tvshow, year, season, episode, set_temp, rar, lang1, lang2, lang3, stack): #standard input
    log(__name__, "Starting search by TV Show")
    # Čisto: Moderni Python (Py3.9+) preferira eksplicitne usporedbe.
    if tvshow is None or tvshow == '':
        log(__name__, "No TVShow name, stop")
        return [], "", ""

    cli = EdnaClient()
    found_tv_shows = cli.search_show(tvshow)
    # Moderni Py3: umjesto __len__() == 0, koristimo not operator
    if not found_tv_shows:
        log(__name__, "TVShow not found, stop")
        return [], "", ""
    # Moderni Py3: umjesto __len__() == 1
    elif len(found_tv_shows) == 1:
        log(__name__, "One TVShow found, auto select")
        tvshow_url = found_tv_shows[0]['url']
    else:
        log(__name__, "More TVShows found, user dialog for select")
        # Uklonjen nekompletan/zakomentiran Kodi/XBMC GUI kod.
        tvshow_url = found_tv_shows[0]['url']
    log(__name__, f"Selected show URL: {tvshow_url}") # Koristimo f-string

    # Konverzija sezone u string jer EdnaClient prima string u list_show_subtitles
    found_season_subtitles = cli.list_show_subtitles(tvshow_url, str(season))

    episode_subtitle_list = None

    for found_season_subtitle in found_season_subtitles:
        # Eksplicitna konverzija u int za usporedbu je sigurnija.
        if found_season_subtitle['episode'] == int(episode) and found_season_subtitle['season'] == int(season):
            episode_subtitle_list = found_season_subtitle
            break

    if episode_subtitle_list is None: # Koristimo "is None" za Py3 standard
        return [], "", ""

    result_subtitles = []
    for episode_subtitle in episode_subtitle_list['versions']:

        result_subtitles.append({
            'filename': episode_subtitle_list['full_title'],
            'link': cli.server_url + episode_subtitle['link'],
            'lang': lng_short2long(episode_subtitle['lang']),
            'rating': "0",
            'sync': False,
            'language_flag': 'flags/' + lng_short2flag(episode_subtitle['lang']) + '.gif',
            'language_name': lng_short2long(episode_subtitle['lang']),
        })

    log(__name__, result_subtitles)

    # Standard output -
    # subtitles list
    # session id (e.g a cookie string, passed on to download_subtitles),
    # message to print back to the user
    # return subtitlesList, "", msg
    return result_subtitles, "", ""


def download_subtitles(subtitles_list, pos, extract_subs, tmp_sub_dir, sub_folder, session_id): #standard input
    selected_subtitles = subtitles_list[pos]

    log(__name__, 'Downloading subtitles')
    
    # Zamijenjeno urllib.urlopen s urllib.request.urlopen (Py3)
    res = urllib.request.urlopen(selected_subtitles['link'])
    
    # Py3: res.info() vraća Message objekt. Konvertiramo ga u string pa ga dekodiramo ako je potrebno.
    # Standardno, Content-Disposition je ASCII, pa str() radi.
    # Zbog potencijalne nekompatibilnosti s Python 3 str(Message), koristimo get() na headeru
    content_disposition = res.info().get('Content-Disposition')
    
    if content_disposition:
        # Re.search vraća MatchObject, group(1) je string (Py3)
        subtitles_filename = re.search('filename="(.+?)"', content_disposition).group(1)
    else:
        # Fallback ako Content-Disposition nije prisutan
        subtitles_filename = os.path.basename(selected_subtitles['link'].split('?')[0])

    log(__name__, f'Filename: {subtitles_filename}')
    
    # subs are in .zip or .rar
    # Re.search vraća string (Py3)
    subtitles_format = re.search(r"\.(\w+?)$", subtitles_filename, re.IGNORECASE).group(1)
    log(__name__, f"Subs in {subtitles_format}")
    
    if subtitles_format.lower() in ['srt', 'sub']: # Koristimo .lower() za sigurnost
        compressed = False
        store_path = os.path.join(sub_folder, subtitles_filename)
        subtitles_file = subtitles_filename # Vraćamo samo ime datoteke, ne cijelu putanju
    else:
        compressed = True
        store_path = extract_subs
        subtitles_file = subtitles_format
    
    # Otvaranje datoteke u binarnom modu za pisanje 'wb'
    with open(store_path, 'wb') as store_path_file:
        # res.read() vraća bytes (Py3)
        store_path_file.write(res.read())
        
    # Standard output -
    # True if the file is packed as zip: addon will automatically unpack it.
    # language of subtitles,
    # Name of subtitles file if not packed (or if we unpacked it ourselves)
    # return False, language, subs_file
    return compressed, selected_subtitles['lang'], subtitles_file


def lng_short2long(lang):
    if lang == 'CZ':
        return 'Czech'
    if lang == 'SK':
        return 'Slovak'
    return 'English'


def lng_long2short(lang):
    if lang == 'Czech':
        return 'CZ'
    if lang == 'Slovak':
        return 'SK'
    return 'EN'


def lng_short2flag(lang):
    return languageTranslate(lng_short2long(lang), 0, 2)


class EdnaClient(object):

    def __init__(self):
        self.server_url = "http://www.edna.cz"

    def search_show(self, title):
        # Zamijenjeno urllib.urlencode s urllib.parse.urlencode (Py3)
        # Py3: urlencode prima string i vraća string
        enc_title = urllib.parse.urlencode({"q": title}) 
        
        # Zamijenjeno urllib.urlopen s urllib.request.urlopen (Py3)
        res = urllib.request.urlopen(self.server_url + "/vyhledavani/?" + enc_title)
        shows = []
        
        if re.search("/vyhledavani/\?q=", res.geturl()):
            log(__name__, "Parsing search result")
            # res.read() vraća bytes. Potrebno je dekodirati u string za regex parsiranje.
            res_body = re.search(r"<ul class=\"list serieslist\">(.+?)</ul>", res.read().decode('utf-8'), re.IGNORECASE | re.DOTALL)
            if res_body:
                for row in re.findall(r"<li>(.+?)</li>", res_body.group(1), re.IGNORECASE | re.DOTALL):
                    show = {}
                    # Dodano 'r' za raw string i promijenjena interna sintaksa regexa za čistoću
                    show_reg_exp = re.compile(r"<h3><a href=\"(.+?)\">(.+?)</a></h3>", re.IGNORECASE | re.DOTALL)
                    # Nema potrebe za .groups() ako se koristi destructuring
                    show['url'], show['title'] = re.search(show_reg_exp, row).groups()
                    shows.append(show)
        else:
            log(__name__, "Parsing redirect to show URL")
            show = {}
            show['url'] = re.search(self.server_url + "(.+)", res.geturl()).group(1)
            show['title'] = title
            shows.append(show)
        return shows

    def list_show_subtitles(self, show_url, show_series):
        # Zamijenjeno urllib.urlopen s urllib.request.urlopen (Py3)
        res = urllib.request.urlopen(self.server_url + show_url + "titulky/?season=" + show_series)
        
        # getcode() je Py3 ekvivalent za getcode()
        if not res.getcode() == 200:
            return []
            
        subtitles = []
        # res.read() vraća bytes. Potrebno je dekodirati u string za regex parsiranje.
        html_content = res.read().decode('utf-8')
        # Dodano 'r' za raw string, korišteno capture grupiranje
        html_subtitle_table = re.search(r"<table class=\"episodes\">.+<tbody>(.+?)</tbody>.+</table>", html_content, re.IGNORECASE | re.DOTALL)
        
        if html_subtitle_table is None: # Koristimo "is None"
            return []
            
        for html_episode in re.findall(r"<tr>(.+?)</tr>", html_subtitle_table.group(1), re.IGNORECASE | re.DOTALL):
            subtitle = {}
            # Učinjena sintaksa za regex čistijom
            show_title_with_numbers = re.sub(r"<[^<]+?>", "", re.search(r"<h3>(.+?)</h3>", html_episode).group(1))
            subtitle['full_title'] = show_title_with_numbers
            # Dodano 'r' za raw string
            show_title_with_numbers = re.search(r"S([0-9]+)E([0-9]+): (.+)", show_title_with_numbers).groups()
            subtitle['season'] = int(show_title_with_numbers[0])
            subtitle['episode'] = int(show_title_with_numbers[1])
            subtitle['title'] = show_title_with_numbers[2]
            subtitle['versions'] = []
            # Dodano 'r' za raw string
            for subs_url, subs_lang in re.findall(r"a href=\"(.+?)\" class=\"flag\".+?><i class=\"flag\-.+?\">(cz|sk)</i>", html_episode):
                subtitle_version = {}
                # Dodano 'r' za raw string
                subtitle_version['link'] = re.sub(r"/titulky/#content", r"/titulky/?direct=1", subs_url)
                subtitle_version['lang'] = subs_lang.upper()
                subtitle['versions'].append(subtitle_version)
            # Moderni Py3: umjesto __len__() > 0
            if subtitle['versions']:
                subtitles.append(subtitle)
        return subtitles
        
        
        