# -*- coding: UTF-8 -*-
# Uklonjen from __future__ import absolute_import
import urllib.parse
import urllib.request
import re
import requests
from ..utilities import log, hashFile, languageTranslate


# Koristimo requests za stabilniji HTTP
cli_session = requests.Session()
# Definiramo server_url izvan klase za SerialZoneClient
SERIALZONE_URL = "http://www.serialzone.cz"


def search_subtitles(file_original_path, title, tvshow, year, season, episode, set_temp, rar, lang1, lang2, lang3, stack): #standard input
    log(__name__, "Starting search by TV Show")
    if (tvshow is None or tvshow == ''):
        log(__name__, "No TVShow name, stop")
        return [], "", ""

    cli = SerialZoneClient(cli_session)
    found_tv_shows = cli.search_show(tvshow)
    
    if (len(found_tv_shows) == 0): # Promijenjeno: __len__() u len()
        log(__name__, "TVShow not found, stop")
        return [], "", ""
    elif (len(found_tv_shows) == 1):
        log(__name__, "One TVShow found, auto select")
        tvshow_url = found_tv_shows[0]['url']
    else:
        log(__name__, "More TVShows found, user dialog for select")
        menu_dialog = []
        for found_tv_show in found_tv_shows:
            if (found_tv_show['orig_title'] == found_tv_show['title']):
                menu_dialog.append(f"{found_tv_show['title']} - {found_tv_show['years']}")
            else:
                menu_dialog.append(f"{found_tv_show['title']} / {found_tv_show['orig_title']} - {found_tv_show['years']}")
        # Dio koda za xbmcgui.Dialog je zakomentiran u originalu,
        # ali moramo osigurati da se tvshow_url postavi. Uzimamo prvi show kao fallback.
        # Pretpostavka: Ako nema korisničkog unosa, uzmi prvi.
        tvshow_url = found_tv_shows[0]['url']
        
    log(__name__, f"Selected show URL: {tvshow_url}")

    try:
        file_size, file_hash = hashFile(file_original_path, rar)
    except:
        file_size, file_hash = -1, None
    log(__name__, f"File size: {file_size}")

    found_season_subtitles = cli.list_show_subtitles(tvshow_url, season)

    episode_subtitle_list = None

    for found_season_subtitle in found_season_subtitles:
        # Pobrinite se da se stringovi pretvore u int za usporedbu
        if (found_season_subtitle['episode'] == int(episode) and found_season_subtitle['season'] == int(season)):
            episode_subtitle_list = found_season_subtitle
            break

    if episode_subtitle_list is None:
        return [], "", ""

    max_down_count = 0
    for episode_subtitle in episode_subtitle_list['versions']:
        if max_down_count < episode_subtitle['down_count']:
            max_down_count = episode_subtitle['down_count']

    log(__name__, f"Max download count: {max_down_count}")

    result_subtitles = []
    # Izbjegavamo dijeljenje s nulom
    rating_divisor = max_down_count if max_down_count > 0 else 1

    for episode_subtitle in episode_subtitle_list['versions']:

        # Korištenje f-stringova
        print_out_filename = f"{episode_subtitle['rip']} by {episode_subtitle['author']}"
        if episode_subtitle['notes'] is not None:
            print_out_filename = f"{print_out_filename} ({episode_subtitle['notes']})"
        
        rating_value = str(int(episode_subtitle['down_count'] * 10 / rating_divisor)) # Osiguravamo int
        
        result_subtitles.append({
            'filename': print_out_filename,
            'link': episode_subtitle['link'],
            'lang': lng_short2long(episode_subtitle['lang']),
            'rating': rating_value,
            'sync': (episode_subtitle['file_size'] == file_size),
            'language_flag': f'flags/{lng_short2flag(episode_subtitle["lang"])}.gif',
            'language_name': lng_short2long(episode_subtitle['lang']),
        })

    log(__name__, f"Found subtitles: {result_subtitles}")

    return result_subtitles, "", ""


def download_subtitles(subtitles_list, pos, zip_subs, tmp_sub_dir, sub_folder, session_id): #standard input

    selected_subtitles = subtitles_list[pos]
    link = selected_subtitles['link']

    log(__name__, f"Selected subtitle link: {link}")

    log(__name__, 'Downloading subtitle zip')
    
    # PROMJENA: Koristimo urllib.request za direktno preuzimanje datoteke jer requests.Session nije definiran globalno ovdje.
    # Ako je sesija neophodna, trebalo bi je proslijediti. Ako ne, koristimo standardni urllib.request.
    try:
        # Preuzimamo bajtove izravno
        with urllib.request.urlopen(link) as res:
            subtitles_data = res.read()
    except Exception as e:
        log(__name__, f"Download failed: {e}")
        return False, selected_subtitles['lang'], "" # Povratak greške

    log(__name__, f'Saving to file {zip_subs}')
    # 'wb' za pisanje bajtova
    try:
        with open(zip_subs, 'wb') as zip_file:
            zip_file.write(subtitles_data)
    except Exception as e:
        log(__name__, f"File saving failed: {e}")
        return False, selected_subtitles['lang'], "" # Povratak greške

    # Standard output -
    # True if the file is packed as zip: addon will automatically unpack it.
    # language of subtitles,
    # Name of subtitles file if not packed (or if we unpacked it ourselves)
    # return False, language, subs_file
    return True, selected_subtitles['lang'], ""


def lng_short2long(lang):
    if lang == 'CZ':
        return 'Czech'
    if lang == 'SK':
        return 'Slovak'
    return 'English'


def lng_long2short(lang):
    if lang == 'Czech':
        return 'CZ'
    if lang == 'Slovak':
        return 'SK'
    return 'EN'


def lng_short2flag(lang):
    # languageTranslate se uvozi iz ..utilities.
    # Ovdje pretpostavljamo da funkcionira ispravno i vraća 2-slovni kod zastavice (npr. 'cz', 'sk')
    return languageTranslate(lng_short2long(lang), 0, 2)


class SerialZoneClient(object):

    # Dodajemo sesiju kao argument za korištenje requests.Session
    def __init__(self, session):
        self.session = session
        self.server_url = SERIALZONE_URL

    def search_show(self, title):
        # Korištenje urllib.parse.urlencode (Py3)
        enc_title = urllib.parse.urlencode({"co": title, "kde": "serialy"})
        
        # Korištenje requests za dohvaćanje, stabilnije za kodiranje/rukovanje greškama
        search_url = f"{self.server_url}/hledani/?{enc_title}"
        
        try:
            res = self.session.get(search_url, timeout=10)
            res.raise_for_status()
            # PROMJENA: Koristimo res.text za dekodirani string
            res_body = res.text
        except requests.RequestException as e:
            log(__name__, f"Search failed for {search_url}: {e}")
            return []

        shows = []
        # Prilagođavanje Regexa za rad s dekodiranim stringom (res.text)
        try:
            # Serialzone koristi Windows-1250, ali requests bi trebao automatski detektirati ili koristiti UTF-8 fallback
            # Pretpostavimo da je res.text ispravno dekodiran.
            match = re.search(r"<div class=\"column4 wd2 fl-left\">(.+?)<div class=\"cl12px fl-left\"></div>", res_body, re.IGNORECASE | re.DOTALL)
            if match:
                res_body = match.group(1)
            else:
                # Ako ne nađemo prvi dio, nastavljamo s cijelim tijelom (fallback)
                log(__name__, "Could not find main search result block.")
                
        except Exception as e:
            log(__name__, f"Regex error in search_show parsing: {e}")
            return []

        for row in re.findall(r'<li>(.+?)</li>', res_body, re.IGNORECASE | re.DOTALL):
            if re.search(r"\/serial\/", row):
                show = {}
                # Korištenje raw stringa i fiksiranje group() poziva
                show_reg_exp = re.compile(r"<a href=\"(.+?)\">(.+?) <span class=\"vysilani\">\((.+?)\)</span></a><br />(.+?)$")
                match = re.search(show_reg_exp, row)
                if match:
                    show['url'], show['title'], show['years'], show['orig_title'] = match.groups()
                    # PROMJENA: fiksiranje entiteta
                    show['years'] = show['years'].replace("&#8211;", "-")
                    shows.append(show)
        return shows

    def list_show_subtitles(self, show_url, show_series):
        # Provjera da li je show_url puni URL
        if not show_url.startswith("http"):
            full_url = f"{self.server_url}{show_url}titulky/{show_series}-rada/"
        else:
            full_url = f"{show_url}titulky/{show_series}-rada/"
            
        log(__name__, f"Fetching subtitles from: {full_url}")
        
        try:
            res = self.session.get(full_url, timeout=10)
            if not res.status_code == 200:
                log(__name__, f"Failed to get subtitles page: Status {res.status_code}")
                return []
            # PROMJENA: Koristimo res.text
            res_body = res.text
        except requests.RequestException as e:
            log(__name__, f"Subtitle list request failed: {e}")
            return []

        subtitles = []
        # Prilagođavanje Regexa za rad s dekodiranim stringom
        # Izdvajanje svih epizoda s titlovima
        episode_blocks = re.findall(r'<div .+? class="sub-line .+?>(.+?)</div></div></div></div>', res_body, re.IGNORECASE | re.DOTALL)
        
        for html_episode in episode_blocks:
            subtitle = {}
            # Dijeljenje bloka epizode na individualne verzije titlova
            subtitle_versions = html_episode.split("<div class=\"sb1\">")
            
            for i, html_subtitle in enumerate(subtitle_versions):
                if i == 0: # Prvi dio bloka sadrži broj epizode
                    show_numbers = re.search(r"<div class=\"sub-nr\">(.+?)</div>", html_subtitle)
                    if show_numbers:
                        # Korištenje raw stringa r""
                        match_nums = re.search(r"([0-9]+)x([0-9]+)", show_numbers.group(1))
                        if match_nums:
                            subtitle['season'], subtitle['episode'] = match_nums.groups()
                            subtitle['season'] = int(subtitle['season'])
                            subtitle['episode'] = int(subtitle['episode'])
                            subtitle['versions'] = []
                        else:
                            # Preskoči ako se brojevi epizoda ne mogu parsirati
                            break
                    else:
                        break # Preskoči ako nema broja epizode
                else: # Ostali dijelovi su pojedinačne verzije titlova
                    subtitle_version = {}
                    try:
                        subtitle_version['lang'] = re.search(r"<div class=\"sub-info-menu sb-lang\">(.+?)</div>", html_subtitle).group(1)
                        subtitle_version['link'] = re.search(r"<a href=\"(.+?)\" .+? class=\"sub-info-menu sb-down\">", html_subtitle).group(1)
                        # Koristimo re.sub za uklanjanje HTML tagova iz autora
                        author_match = re.search(r"<div class=\"sub-info-auth\">(.+?)</div>", html_subtitle)
                        subtitle_version['author'] = re.sub(r"<[^<]+?>", "", author_match.group(1)) if author_match else "N/A"
                        
                        subtitle_version['rip'] = re.search(r"<div class=\"sil\">Verze / Rip:</div><div class=\"sid\"><b>(.+?)</b>", html_subtitle).group(1)
                        
                        try:
                            note_match = re.search(r"<div class=\"sil\">Poznámka:</div><div class=\"sid\">(.*?)</div>", html_subtitle)
                            subtitle_version['notes'] = note_match.group(1).strip() if note_match else None
                            if subtitle_version['notes'] == '':
                                subtitle_version['notes'] = None
                        except:
                            subtitle_version['notes'] = None
                            
                        # Početni int() se izvodi na match.group(1)
                        down_count_match = re.search(r"<div class=\"sil\">Počet stažení:</div><div class=\"sid2\">(.+?)x</div>", html_subtitle)
                        subtitle_version['down_count'] = int(down_count_match.group(1)) if down_count_match else 0
                        
                        try:
                            # Fiksiramo regex za file_size
                            size_match = re.search(r"<span class=\"fl-right\" title=\".+\">\((.+?) b\)</span>", html_subtitle)
                            if size_match:
                                file_size_str = size_match.group(1)
                                # Uklanjamo razmake prije pretvaranja u int
                                subtitle_version['file_size'] = int(file_size_str.replace(" ", ""))
                            else:
                                subtitle_version['file_size'] = -1
                        except:
                            subtitle_version['file_size'] = -1
                            
                        subtitle['versions'].append(subtitle_version)

                    except Exception as e:
                        # Logirajte grešku parsiranja pojedinačne verzije titla
                        log(__name__, f"Error parsing subtitle version: {e}")
                        continue # Nastavite s sljedećom verzijom

            if 'versions' in subtitle and subtitle['versions']:
                subtitles.append(subtitle)
        return subtitles
        
        
        